//
// This is a minimal Rust program for demonstrating the usage of the Goodtime Time Tracking REST-API v2.1
//

use reqwest::Error;
use reqwest::{Client, header};

async fn hello(client: &Client) -> Result<(), Error> {
    let url = "https://getgoodtime.com/goodtime-rs/api/hello/sayhello";
    let response = client.get(url).send().await?;
    println!("hello: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn login(client: &Client) -> Result<String, Box<dyn std::error::Error>> {
    let url = "https://getgoodtime.com/goodtime-rs/api/login";
    let credentials = serde_json::json!({
        "loginemail": "admin@somedomain.com",
        "apipassword": "12345678"
    });
    let response = client.post(url).json(&credentials).send().await?;
    println!("login: {}", response.status());
    let token = response
        .headers()
        .get(header::AUTHORIZATION)
        .ok_or("Authorization header missing")?
        .to_str()?
        .to_string();

    let company_id = response.text().await?;
    println!("company_id: {}", company_id);
    println!("token: {}", token);
    Ok(token)
}

async fn secured_hello(client: &Client, token: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = "https://getgoodtime.com/goodtime-rs/api/hello/securedSayhello";
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.get(url).headers(headers).send().await?;
    println!("secured_hello: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn get_timetypes(client: &Client, token: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = "https://getgoodtime.com/goodtime-rs/api/time-v2.1/timetypes";
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.get(url).headers(headers).send().await?;
    println!("get_timetypes: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn get_users(client: &Client, token: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = "https://getgoodtime.com/goodtime-rs/api/time-v2.1/users";
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.get(url).headers(headers).send().await?;
    println!("get_users: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn get_projects(client: &Client, token: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = "https://getgoodtime.com/goodtime-rs/api/time-v2.1/projects";
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.get(url).headers(headers).send().await?;
    println!("get_projects: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn get_projects_for_user(client: &Client, token: &str, user_email: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = format!("https://getgoodtime.com/goodtime-rs/api/time-v2.1/projects/{}", user_email);
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.get(&url).headers(headers).send().await?;
    println!("get_projects_for_user: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn get_timeentries_for_user(client: &Client, token: &str, user_email: &str, start: &str, end: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = format!(
        "https://getgoodtime.com/goodtime-rs/api/time-v2.1/timeentries?useremail={}&start={}&end={}",
        user_email, start, end
    );
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.get(&url).headers(headers).send().await?;
    println!("get_timeentries_for_user: {}", response.status());
    let body = response.text().await?;
    println!("{}", body);
    Ok(())
}

async fn create_timeentry(client: &Client, token: &str) -> Result<String, Box<dyn std::error::Error>> {
    let url = "https://getgoodtime.com/goodtime-rs/api/time-v2.1/createtimeentry";
    let param = serde_json::json!({
        "useremail": "admin@somedomain.com",
        "starttime": "2022-12-30T10:24:00",
        "endtime": "2022-12-30T10:27:00",
        "timetypeid": "004c396a-9e24-411c-b6fd-9250a169b831",
        "projectid": "99f5012f-3ead-491c-ae91-28e3a2fe7407",
        "comment": "my comment"
    });

    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.post(url).json(&param).headers(headers).send().await?;
    let body = response.text().await?;
    println!("create_timeentry: {}", body);
    // we return the timeitem_id as String
    let v: Vec<String> = serde_json::from_str(&body)?;
    let timeitem_id = v[0].clone();
    Ok(timeitem_id)
}

async fn delete_timeentry(client: &Client, token: &str, timeitem_id: &str) -> Result<(), Box<dyn std::error::Error>> {
    let url = format!("https://getgoodtime.com/goodtime-rs/api/time-v2.1/deletetimeentry/{}", timeitem_id);
    let mut headers = header::HeaderMap::new();
    headers.insert(header::AUTHORIZATION, token.parse()?);

    let response = client.delete(&url).headers(headers).send().await?;
    println!("delete_timeentry: {}", response.status());
    Ok(())
}

#[tokio::main(flavor = "current_thread")]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    let client = Client::new();

    hello(&client).await?;
    let token = login(&client).await?;
    secured_hello(&client, &token).await?;
    get_timetypes(&client, &token).await?;
    get_users(&client, &token).await?;
    get_projects(&client, &token).await?;
    get_projects_for_user(&client, &token, "admin@somedomain.com").await?;
    get_timeentries_for_user(&client, &token, "admin@somedomain.com", "2020-01-01T00:00:00", "2020-01-07T23:59:00").await?;
    let timeitem_id = create_timeentry(&client, &token).await?;
    delete_timeentry(&client, &token, &timeitem_id).await?;

    Ok(())
}

